unit UPosInterface;

interface
uses
  Classes;

type
  TResponseType = (rtPayment, rtBillPayment);

  TPosEventArgs = class
  private
    FResponse: String;
    FResponseType: TResponseType;
    Private FResponseList: TStringList;

    procedure SetResponse(aResponse: String);
    function GetResponseAt(aIndex: Integer): String;
    function GetResponsePrefix: String;
    function GetErrorCode: Integer;
    function GetErrorMsg: String;
    function GetPaymentAmount: String;
    function GetRRN: String;
    function GetStan: String;
    function GetDateTime: String;
    function GetMerchantId: String;
    function GetTerminalId: String;
    function GetCardNumber: String;
    function GetMessageId: String;
    function GetBillId: String;
    function GetPaymentId: String;
    function GetIsValid: Boolean;
  Public
    constructor Create(AResponseType: TResponseType);
    destructor Destroy; override;
    function ToString: String;

    property Response: String read FResponse write SetResponse;
    property ResponsePrefix: String read GetResponsePrefix;
    property ErrorCode: Integer read GetErrorCode;
    property ErrorMsg: String read GetErrorMsg;
    property PaymentAmount: String read GetPaymentAmount;
    property RRN: String read GetRRN;
    property Stan: String read GetStan;
    property DateTime: String read GetDateTime;
    property MerchantId: String read GetMerchantId;
    property TerminalId: String read GetTerminalId;
    property CardNumber: String read GetCardNumber;
    property MessageId: String read GetMessageId;
    property BillId: String read GetBillId;
    property PaymentId: String read GetPaymentId;
    property IsValid: Boolean read GetIsValid;
  end;

  //TNotifyEvent = procedure(Sender: TObject) of object;
  TOnPaymentResponse = procedure(sender: TObject; e: TPosEventArgs) of object;

  EnumResponseResult =
  (
    PAY_SUCCESS
    , ERR_PAY_INEXECUTION
    , ERR_PAY_INEQUALPAYMENTANDBILLS
    , ERR_PAY_OUTOFRANGEBILLCOUNT
    , ERR_PAY_OUTOFRANGESHEBACOUNT
    , ERR_PAY_INVALIDPAYMENTSUM
    , ERR_PAY_TIMEOUT
    , ERR_PAY_CANCELED

    , ERR_CLIENTSOCKET_INIT
    , ERR_CLIENTSOCKET_CONNECT
    , ERR_CLIENTSOCKET_SEND

    , ERR_SERVERSOCKET_INIT
    , ERR_SERVERSOCKET_CREATE
    , ERR_SERVERSOCKET_SEND
  );


  PPosInterface = ^TPosInterface;
  TPosInterface = class
  private
    FMessageID: Integer;
    FIPAddress: String;
    FPortNo: Integer;
    FOnPaymentResponse: TOnPaymentResponse;
    FOnBillPaymentResponse: TOnPaymentResponse;

    function GetIsExecuted(): Boolean;
    function GetPercentElapsedTime(): Integer;
  public
    procedure CallbackPayment(aResponse: String);
    procedure CallbackBillPayment(aResponse: String);
    function DoBillPayment(aPaymentIds: String; aBillIds: String; aTimeout: Integer): Integer;
    function DoPayment(aPaymentAmount: Integer; aTashim, aInvoiceNumber: String; aTimeout: Integer): Integer;
    function GetResponseErrorMessage(aResponseResult: Integer): String;
    procedure CancelExecution();

  public
    property MessageID: Integer read FMessageID write FMessageID;
    property IPAddress: String read FIPAddress write FIPAddress;
    property PortNo: Integer read FPortNo write FPortNo default 17000;
    property OnPaymentResponse: TOnPaymentResponse read FOnPaymentResponse write FOnPaymentResponse;
    property OnBillPaymentResponse: TOnPaymentResponse read FOnBillPaymentResponse write FOnBillPaymentResponse;
    property IsExecuted: Boolean read GetIsExecuted;
    property PercentElapsedTime: Integer read GetPercentElapsedTime;
  end;


implementation
uses
  SysUtils, Dialogs;

function Split(Str: string; SubStr: string): TStringList;
var
  I: Integer;
  S, Tmp: string;
  List: TStringList;
begin
  List := TStringList.Create;
  S := Str;
  while Length(S) > 0 do
  begin
    I := Pos(SubStr, S);
    if I = 0 then
    begin
      List.Add(S);
      S := '';
    end
    else
    begin
      if I = 1 then
      begin
        List.Add('');
        Delete(S, 1, Length(SubStr));
      end
      else
      begin
        Tmp := S;
        Delete(Tmp, I, Length(Tmp));
        List.Add(Tmp);
        Delete(S, 1, I + Length(SubStr) - 1);
        if Length(S) = 0 then
          List.Add('');
      end;
    end;
  end;
  Result := List;
end;


{ TPosEventArgs }
constructor TPosEventArgs.Create(AResponseType: TResponseType);
begin
  inherited Create;
  FResponseType := AResponseType;
end;

destructor TPosEventArgs.Destroy;
begin
  if Assigned(FResponseList) then
    FResponseList.Free;

  inherited;
end;

procedure TPosEventArgs.SetResponse(aResponse: String);
begin
  FResponse := aResponse;
  FResponseList := Split(FResponse, ',');
end;

function TPosEventArgs.GetResponseAt(aIndex: Integer): String;
begin
  if ((aIndex >= 0) and (aIndex < FResponseList.Count)) then
    Result := FResponseList[aIndex]
  else
    Result := '';
end;

function TPosEventArgs.GetResponsePrefix: String;
begin
  Result := GetResponseAt(0);
end;

function TPosEventArgs.GetErrorCode: Integer;
begin
  Result := StrToInt(GetResponseAt(1));
end;

function TPosEventArgs.GetErrorMsg: String;
begin
  case GetErrorCode of
    0:
      Result := 'ǘ ';
    3:
      Result := ' Ԑ ';
    777:
      Result := '    ';
    5:
      Result := '  ǘ   ';
    6:
      Result := ' ';
    12:
      Result := 'ǘ ';
    51:
      Result := '   ';
    55:
      Result := '   ';
    56:
      Result := '  ';
    61:
      Result := ' ǘ      ';
    84:
      Result := '     ';
    400:
      Result := '      ';
    402:
      Result := '    ';
    403:
      Result := '  ';
    410:
      Result := 'ǁ   ';
    412:
      Result := '     ';
    500:
      Result := '  ';
    501:
      Result := '     :ǐ   ѐ';
  else
      Result := 'ǘ ';
  end;
end;

function TPosEventArgs.GetPaymentAmount: String;
begin
  Result := GetResponseAt(2);
end;

function TPosEventArgs.GetRRN: String;
begin
  Result := GetResponseAt(3);
end;

function TPosEventArgs.GetStan: String;
begin
  Result := GetResponseAt(4);
end;

function TPosEventArgs.GetDateTime: String;
begin
  Result := GetResponseAt(5);
end;

function TPosEventArgs.GetMerchantId: String;
begin
  Result := GetResponseAt(6);
end;

function TPosEventArgs.GetTerminalId: String;
begin
  Result := GetResponseAt(7);
end;

function TPosEventArgs.GetCardNumber: String;
begin
  Result := GetResponseAt(8);
end;

function TPosEventArgs.GetMessageId: String;
begin
  Result := GetResponseAt(9);
end;

function TPosEventArgs.GetBillId: String;
begin
  if FResponseType = rtBillPayment then
    Result := GetResponseAt(10)
  else
    Result := '';
end;

function TPosEventArgs.GetPaymentId: String;
begin
  if FResponseType = rtBillPayment then
    Result := GetResponseAt(11)
  else
    Result := '';
end;

function TPosEventArgs.GetIsValid: Boolean;
begin
  Result := (((FResponseType = rtPayment)
                  and (GetResponseAt(0) = '$POSBUY'))
                  or ((FResponseType = rtBillPayment)
                  and (GetResponseAt(0) = '$POSBIL')));
end;

function TPosEventArgs.ToString: String;
begin
  Result := '';
  Result := Result + 'Response = ' + ResponsePrefix;
  Result := Result + ';ErrorCode = ' + IntToStr(ErrorCode);
  Result := Result + ';ErrorMsg = ' + ErrorMsg;
  Result := Result + ';PaymentAmount = ' + PaymentAmount;
  Result := Result + ';RRN = ' + RRN;
  Result := Result + ';Stan = ' + Stan;
  Result := Result + ';DateTime = ' + DateTime;
  Result := Result + ';MerchantId = ' + MerchantId;
  Result := Result + ';TerminalId = ' + TerminalId;
  Result := Result + ';CardNumber = ' + CardNumber;
  Result := Result + ';MessageId = ' + MessageId;
  if FResponseType = rtBillPayment then
  begin
    Result := Result + ';BillId = ' + BillId;
    Result := Result + ';PaymentId = ' + PaymentId;
  end;
end;


const
{ $DEFINE Release}    //  for release use this {$DEFINE Release}

{$IFDEF Release}
  DLLPATH = 'PcPos.dll';
{$ELSE}
  DLLPATH = '..\PcPos\Debug\PcPos.dll';
{$ENDIF}

type
  PResponseCallback = ^TResponseCallback;
  TResponseCallback = procedure(aResponse: PChar; aCallbackParam: PPosInterface); stdcall;

function Pos_BillPayment(aMessageID: Integer; aPaymentIds, aBillIds: PChar;
  aDateTime, aIPAdress: PChar;
  aPortNo, aTimeout: Integer; var aSocketLastError: Integer;
  aCallbackFunc: TResponseCallback; aCallbackParam: PPosInterface): Integer;
  stdcall; external DLLPATH name 'Pos_BillPayment';

function Pos_Payment(aMessageID: Integer; aPaymentAmount: Integer; aTashim, aInvoiceNumber: PChar;
  aDateTime, aIPAdress: PChar;
  aPortNo, aTimeout: Integer; var aSocketLastError: Integer;
  aCallbackFunc: TResponseCallback; aCallbackParam: PPosInterface): Integer;
  stdcall; external DLLPATH name 'Pos_Payment';

function Pos_IsExecuted: Integer; stdcall; external DLLPATH name 'Pos_IsExecuted';
function Pos_CancelExecution: Integer; stdcall; external DLLPATH name 'Pos_CancelExecution';
function Pos_PercentElapsedTime: Integer; stdcall; external DLLPATH name 'Pos_PercentElapsedTime';

procedure Callback_Payment(aResponse: PChar; aCallbackParam: PPosInterface); stdcall;
var
  vResponse: String;
begin
  vResponse := aResponse;
  aCallbackParam^.CallbackPayment(vResponse);
end;

procedure Callback_BillPayment(aResponse: PChar; aCallbackParam: PPosInterface); stdcall;
var
  vResponse: String;
begin
  vResponse := aResponse;
  aCallbackParam^.CallbackBillPayment(vResponse);
end;


{ TPosInterface }
procedure TPosInterface.CallbackPayment(aResponse: String);
var
  vPosEventArgs: TPosEventArgs;
begin
  if Assigned(FOnPaymentResponse) then
  begin
    vPosEventArgs := TPosEventArgs.Create(rtPayment);
    vPosEventArgs.Response := aResponse;
    FOnPaymentResponse(Self, vPosEventArgs);
    vPosEventArgs.Free;
  end;
end;

procedure TPosInterface.CallbackBillPayment(aResponse: String);
var
  vPosEventArgs: TPosEventArgs;
begin
  if Assigned(FOnBillPaymentResponse) then
  begin
    vPosEventArgs := TPosEventArgs.Create(rtBillPayment);
    vPosEventArgs.Response := aResponse;
    FOnBillPaymentResponse(Self, vPosEventArgs);
    vPosEventArgs.Free;
  end;
end;

function TPosInterface.GetIsExecuted(): Boolean;
begin
  Result := Pos_IsExecuted() <> 0;
end;

function TPosInterface.GetPercentElapsedTime(): Integer;
begin
  Result := Pos_PercentElapsedTime();
end;

procedure TPosInterface.CancelExecution();
begin
  Pos_CancelExecution
end;

function TPosInterface.GetResponseErrorMessage(aResponseResult: Integer): String;
var
  vMsg: String;
begin
{
  if (aResponseResult = 0) then
    Result := ''
  else
    Result := IntToStr(aResponseResult);
}
   case EnumResponseResult(aResponseResult) of
      PAY_SUCCESS:
        vMsg := '';
      ERR_PAY_INEXECUTION:
        vMsg := 'ERR_PAY_INEXECUTION';
      ERR_PAY_INEQUALPAYMENTANDBILLS:
        vMsg := 'ERR_PAY_INEQUALPAYMENTANDBILLS';
      ERR_PAY_OUTOFRANGEBILLCOUNT:
        vMsg := 'ERR_PAY_OUTOFRANGEBILLCOUNT';
      ERR_PAY_OUTOFRANGESHEBACOUNT:
        vMsg := 'ERR_PAY_OUTOFRANGESHEBACOUNT';
      ERR_PAY_INVALIDPAYMENTSUM:
        vMsg := 'ERR_PAY_INVALIDPAYMENTSUM';
      ERR_PAY_TIMEOUT:
        vMsg := 'ERR_PAY_TIMEOUT';
      ERR_PAY_CANCELED:
        vMsg := 'ERR_PAY_CANCELED';

      ERR_CLIENTSOCKET_INIT:
        vMsg := 'ERR_CLIENTSOCKET_INIT';
      ERR_CLIENTSOCKET_CONNECT:
        vMsg := 'ERR_CLIENTSOCKET_CONNECT';
      ERR_CLIENTSOCKET_SEND:
        vMsg := 'ERR_CLIENTSOCKET_SEND';

      ERR_SERVERSOCKET_INIT:
        vMsg := 'ERR_SERVERSOCKET_INIT';
      ERR_SERVERSOCKET_CREATE:
        vMsg := 'ERR_SERVERSOCKET_CREATE';
      ERR_SERVERSOCKET_SEND:
        vMsg := 'ERR_SERVERSOCKET_SEND';
      else
        vMsg := 'Error: ' + IntToStr(aResponseResult);
    end;
    Result := vMsg;
end;

function TPosInterface.DoBillPayment(aPaymentIds: String; aBillIds: String; aTimeout: Integer): Integer;
var
  vSocketLastError: Integer;
  vResult: Integer;
  vNow: TDateTime;
  vDateTime: String;
begin
  if IPAddress = '' then raise Exception.Create('IPAddress not initialized');
  if PortNo = 0 then raise Exception.Create('Port number not initialized');

  Inc(FMessageID);

  vSocketLastError := 0;
  vNow := SysUtils.Date + GetTime;
  vDateTime := FormatDateTime('yyyyMMddhhmmss', vNow);
  vResult := Pos_BillPayment(MessageID, PAnsiChar(aPaymentIds), PAnsiChar(aBillIds), PAnsiChar(vDateTime)
    , PAnsiChar(Self.IPAddress), Self.PortNo, aTimeout, vSocketLastError
    , Callback_BillPayment, @Self);
  Result := vResult;
end;

function TPosInterface.DoPayment(aPaymentAmount: Integer; aTashim, aInvoiceNumber: String; aTimeout: Integer): Integer;
var
  vSocketLastError: Integer;
  vResult: Integer;
  vNow: TDateTime;
  vDateTime: String;
begin

  if IPAddress = '' then raise Exception.Create('IPAddress not initialized');
  if PortNo = 0 then raise Exception.Create('Port number not initialized');

  Inc(FMessageID);

  vSocketLastError := 0;
  vNow := SysUtils.Date + GetTime;
  vDateTime := FormatDateTime('yyyyMMddhhmmss', vNow);
  vResult := Pos_Payment(MessageID, aPaymentAmount, PAnsiChar(aTashim), PAnsiChar(aInvoiceNumber), PAnsiChar(vDateTime)
    , PAnsiChar(Self.IPAddress), Self.PortNo, aTimeout, vSocketLastError
    , Callback_Payment, @Self);
  Result := vResult;
end;

end.
